# -*- coding: cp1252 -*-
overrideIcon= "r_arnold_denoice"

######################################################################
#
# Name:  Arnold Denoise
# Author:  Royal Render, Paolo Acampora, Binary Alchemy
# Version:
# Description:
#   Create jobs for Arnold Denoise execution.
#
#
# Copyright (c) Holger Schoenberger - Binary Alchemy
######################################################################

import os


class DenoiseEntry(object):
    """Store info about the arnold job and generate denoise job from UI data"""
    def __init__(self, job_selected_idx, checkbox_name, browser_name):
        """

        :param job_selected_idx: index of the arnold job, to use with rr.jobSelected_get(idx). -1 if none
        :param checkbox_name: checkbox element of the UI for this job, "" if none
        :param browser_name: filename element of the UI for this job
        """
        self._job_sel_idx = job_selected_idx
        self._checkbox_name = checkbox_name
        self._browser_name = browser_name

        self.job = None

    def job_from_ui(self, rr_ui, out_template, render_app):
        """Create job if source is checked in the script UI, return success status.
         The job is not yet added to the joblist

        :param rr_ui: script ui
        :param out_template: output file template
        :param render_app: job render app (Arnold Denoise)
        :return: success status
        """
        if self._checkbox_name and rr_ui.value(self._checkbox_name) == 0:
            return False

        main_sequence = rr_ui.text(self._browser_name)

        self.job = rr.getNewJob()
        self.job.overrideIcon = overrideIcon
        self.job.sceneName = main_sequence

        self.job.imageFileName = out_template
        self.job.imageExtension = ".exr"
        self.job.splitImageFileInto_DirFileExt(False)
        self.job.uiIsChecked = True

        if self._job_sel_idx == -1:
            return True

        arnold_job = rr.jobSelected_get(self._job_sel_idx)
        self.job.waitForPreID_set(0, arnold_job.preID)
        self.job.seqStart = arnold_job.seqStart
        self.job.seqEnd = arnold_job.seqEnd
        self.job.seqStep = arnold_job.seqStep
        self.job.imageWidth = arnold_job.imageWidth
        self.job.imageHeight = arnold_job.imageHeight
        self.job.camera = arnold_job.camera
        self.job.layer = arnold_job.layer
        
        render_app.setVersionBoth(arnold_job.renderApp.rendererVersion)
        self.job.renderApp = render_app
        #rrGlobal.writeLog2(rrGlobal.logLvL.warning, "version old: "+arnold_job.renderApp.rendererVersion+ "   new: "+str(self.job.renderApp.version)+"  "+ self.job.renderApp.versionMinorDisplay(False))

        # disable crop exr on arnold job, or noice won't find the variance info
        self.job.customDataAppend_Str("rrSubmitterParameter", " ApproveJob=1~0")
        self.job.customDataAppend_Str("rrSubmitterParameter", " RenderPreviewFirst=0~0")
        self.job.customDataAppend_Str("NoSceneExistCheck", "true")
        
        arnold_job.customDataAppend_Str("rrSubmitterParameter", " CropEXR=0~0")
        rr.jobSelected_set(self._job_sel_idx, arnold_job)

        self.job.maxWaitForPreID += 1

        return True


def show_denoise_dialog(first_job=0, num_jobs=-1, max_rows=0):
    """Display input dialog for denoise job. Return denoise job entries and last processed index (exclusive)

    :param first_job: index of the job from which to start
    :param num_jobs: number of jobs to process
    :param max_rows: maximum number of job rows
    :return: (denoise_entries[], job_index)
    """

    myUI= rrGlobal.getGenericUI()

    myUI.addItem(rrGlobal.genUIType.label, "infoLabel", "")
    myUI.setText("infoLabel",
                 "Please specify a frame of the sequence to denoise."
                 "\nRequired passes: RGBA, N, Z; Required filter: variance_filter" + " "*60)

    denoise_entries = []

    job_row_base = "SeqLayout_{0}"
    job_seq_base = "MainSequence_{0}"
    job_box_base = "jobCheckbox_{0}"

    if num_jobs < 0:
        num_jobs = rr.jobSelected_count()

    for job_idx in range(first_job, num_jobs):
        a_job = rr.jobSelected_get(job_idx)

        if a_job.imageSingleOutputFile:
            continue

        r_app = a_job.renderApp

        if r_app.rendererName == "DeNoice":
            continue
        if a_job.imageExtension.lower() != ".exr":
            continue
        if not ( "arnold" in r_app.name.lower() or "arnold" in r_app.rendererName.lower() ):
            continue

        # we need preIDs so we can wait for jobs
        if a_job.preID == 0:
            rr.populatePreIDs()

        arnold_out_dir = a_job.imageDir
        arnold_out_file = "".join(
            (a_job.imageFileName, str(int(a_job.seqStart)).zfill(a_job.imageFramePadding), a_job.imageExtension)
        )
        arnold_out_path = os.path.join(arnold_out_dir, arnold_out_file)
        arnold_out_path = a_job.replaceJobVariables(arnold_out_path, True)

        job_row_name = job_row_base.format(job_idx)
        job_seq_name = job_seq_base.format(job_idx)
        job_box_name = job_box_base.format(job_idx)

        myUI.addItem(rrGlobal.genUIType.layoutH, job_row_name, "")
        myUI.addItem(rrGlobal.genUIType.checkBox, job_box_name, job_row_name)
        myUI.addItem(rrGlobal.genUIType.browseFile, job_seq_name, job_row_name)

        myUI.setText(job_box_name, "")
        myUI.setValue(job_box_name, 1)
        myUI.setText(job_seq_name, arnold_out_path)

        denoise_entries.append(DenoiseEntry(job_idx, job_box_name, job_seq_name))

        if max_rows and len(denoise_entries) > max_rows + 1:
            # check needs to happen here: not all jobs are added
            break

    if denoise_entries:
        # last entry: blank line
        job_row_name = job_row_base.format("blank")
        job_seq_name = job_seq_base.format("blank")
        job_box_name = job_box_base.format("blank")

        myUI.addItem(rrGlobal.genUIType.layoutH, job_row_name, "")
        myUI.addItem(rrGlobal.genUIType.checkBox, job_box_name, job_row_name)
        myUI.addItem(rrGlobal.genUIType.browseFile, job_seq_name, job_row_name)
        myUI.setText(job_box_name, "")
        denoise_entries.append(DenoiseEntry(-1, job_box_name, job_seq_name))
    else:
        # ask user for input
        myUI.addItem(rrGlobal.genUIType.label, "labelMainSequence", "")
        myUI.setText("labelMainSequence", "Image sequence:")
        myUI.addItem(rrGlobal.genUIType.browseFile, "MainSequence", "")
        myUI.addItem(rrGlobal.genUIType.spacerV, "spacerF", "")

        denoise_entries.append(DenoiseEntry(-1, "", "MainSequence"))

    myUI.addItem(rrGlobal.genUIType.spacerV, "spacerF", "")
    myUI.addItem(rrGlobal.genUIType.label, "labelOutFormat", "")
    myUI.setText("labelOutFormat", "Output format:")
    myUI.addItem(rrGlobal.genUIType.browseFile, "OutFormat", "")
    myUI.addItem(rrGlobal.genUIType.spacerV, "spacerF", "")

    myUI.setText("OutFormat", os.path.join("<SceneFolder>", "denoised", "<SceneName>" + "_denoised_" + "####"))

    myUI.addItem(rrGlobal.genUIType.spacerV,"spacerZ","")
    myUI.addItem(rrGlobal.genUIType.layoutH,"btnLayout","")
    myUI.addItem(rrGlobal.genUIType.closeButton,"Create Job", "btnLayout")
    myUI.addItem(rrGlobal.genUIType.closeButton,"Abort","btnLayout")

    job_idx += 1
    if job_idx < num_jobs:
        multi_pages = True
        myUI.addItem(rrGlobal.genUIType.checkBox, "displayNextJobs", "btnLayout")
        myUI.setText("displayNextJobs", "display other jobs")
    else:
        multi_pages = False

    myUI.execute()
    if myUI.value("Create Job") == 0:
        if multi_pages and myUI.value("displayNextJobs") == 1:
            del myUI
            return [], job_idx
        else:
            del myUI
            raise rrCleanExit()

    render_app = rrJob._RenderAppBasic()
    render_app.clear()
    render_app.name = "Arnold"
    render_app.rendererName = "DeNoice"
    render_app.type = "Comp"

    out_format = myUI.text("OutFormat")
    for entry in denoise_entries:
        entry.job_from_ui(myUI, out_format, render_app)
    del myUI

    return denoise_entries, job_idx


def add_denoise_jobs(denoise_entries):
    """Add jobs contained in denoise_entries to the joblist

    :param denoise_entries: list of entries, each entry must have an attribute named 'job'
    """
    new_job_idx = 999  # If the index of the job is higher than the number of jobs,
    # then a new job is added to the end of the job list. So we use 999.

    for entry in denoise_entries:
        job_count = rr.jobAll_count()
        if entry.job:
            # at this point all jobs should have preIDs
            # perhaps a greater value would give us some slack
            # NOTE: preIDs start from 1
            job_count += 1
            entry.job.preID = job_count
            rr.jobAll_set(new_job_idx, entry.job)


if __name__ == "__main__":
    num_rows = 8
    num_jobs = rr.jobSelected_count()

    denoise_entries, job_idx = show_denoise_dialog(max_rows=num_rows)
    while job_idx < num_jobs:
        more_entries, job_idx = show_denoise_dialog(first_job=job_idx, max_rows=num_rows)
        denoise_entries.extend(more_entries)

    add_denoise_jobs(denoise_entries)
